NAVCollection.prototype.analyzeShardKey = function (key, options) {
    if (!isObject(key))
        nav_throwError('argument for analyzeShardKey should be an object');

    if (isUndefined(key.key))
        nav_throwError('shard key for analyzeShardKey is empty');
        
    if (isUndefined(options))
        options = {};

    var fullName = this._db.getName() + "." + this.getName();
    return this._db.adminCommand({ analyzeShardKey: fullName, key, ...options});
};

NAVCollection.prototype.aggregate = function (pipeline, options) {
    if (!Array.isArray(pipeline)) {
        pipeline = Array.from(arguments);
        options = {};
    }
    
    if (!Array.isArray(pipeline))
        nav_throwError("invalid argument");

    if (isUndefined(options))
        options = {};
    
    if (!isObject(options))
        nav_throwError("invalid argument");

    var tmpOptions = Object.extend({}, (options || {}));

    if ('batchSize' in tmpOptions) {
        if (tmpOptions.cursor == null)
            tmpOptions.cursor = {};

        tmpOptions.cursor.batchSize = tmpOptions['batchSize'];
        delete tmpOptions['batchSize'];
    } else if ('useCursor' in tmpOptions) {
        if (tmpOptions.cursor == null)
            tmpOptions.cursor = {};

        delete tmpOptions['useCursor'];
    }

    if (!("cursor" in tmpOptions))
        tmpOptions.cursor = {};

    var maxAwaitTimeMS = tmpOptions.maxAwaitTimeMS;
    delete tmpOptions.maxAwaitTimeMS;

    var commandObj = { aggregate : this.getName(), pipeline : pipeline};
    Object.extend(commandObj, tmpOptions);

    var hasOutStage = pipeline.length >= 1 && pipeline[pipeline.length - 1].hasOwnProperty("$out");

    var batchSize = tmpOptions.cursor ? tmpOptions.cursor.batchSize : undefined;

    var isExplain = tmpOptions.explain ? tmpOptions.explain : false;

    var isWatch = false;
    if (pipeline.length > 0)
        isWatch = pipeline[0].$changeStream ? true : false;

    return this.forwardToCustomFunction("collectionAggregate", hasOutStage, commandObj, { pipeline: pipeline }, tmpOptions, batchSize, isExplain, maxAwaitTimeMS, isWatch);
};

NAVCollection.prototype.addIdIfNeeded = function(obj) {
    if (!isObject(obj))
        nav_throwError('argument for addIdIfNeeded should be an object');

    if (isUndefined(obj._id) && !Array.isArray(obj)) {
        var objOld = obj;
        obj = {_id: new ObjectId()};
        Object.extend(obj, objOld);
    }

    return obj;
};

NAVCollection.prototype._isAcknowledged = function () {
    return this.forwardToCustomFunction("collectionIsAcknowledged");
};

NAVCollection.prototype._getWriteConcern = function () {
    return this.forwardToCustomFunction("collectionGetWriteConcern");
};

NAVCollection.prototype._extractWriteConcern = function (options) {
    var writeConcern = options.writeConcern || this._getWriteConcern();
    
    var writeConcernOptions = ['w', 'wtimeout', 'j', 'fsync'];
    if (options.w != null || options.wtimeout != null || options.j != null ||
        options.fsync != null) {
        writeConcern = {};

        writeConcernOptions.forEach(function (wc) {
            if (options[wc] != null) {
                writeConcern[wc] = options[wc];
            }
        });
    }

    return writeConcern;
};

NAVCollection.prototype.bulkWrite = function (operations, options) {
    options = Object.extend({}, options || {});
    var ordered = (isBoolean(options.ordered)) ? options.ordered : true;

    var writeConcern = this._extractWriteConcern(options);

    var acknowledged = true;
    if (writeConcern && isObject(writeConcern) && writeConcern.w == 0)
        acknowledged = false;

    var result = {acknowledged: acknowledged};

    var bulkOperation = ordered ? this.initializeOrderedBulkOp() : this.initializeUnorderedBulkOp();

    var insertedIds = {};

    operations.forEach(function(operation, index) {
        if (operation.insertOne) {
            if (!operation.insertOne.document)
                nav_throwError('insertOne.document for bulkWrite operation is required');

            operation.insertOne.document = this.addIdIfNeeded(operation.insertOne.document);
            insertedIds[index] = operation.insertOne.document._id;
            bulkOperation.insert(operation.insertOne.document);
        } else if (operation.updateOne) {
            if (!operation.updateOne.filter)
                nav_throwError('updateOne.filter for bulkWrite operation is required');

            if (!operation.updateOne.update)
                nav_throwError('updateOne.update for bulkWrite operation is required');

            var updateOperation = bulkOperation.find(operation.updateOne.filter);
            if (operation.updateOne.upsert)
                updateOperation = updateOperation.upsert();

            if (operation.updateOne.hint)
                updateOperation.hint(operation.updateOne.hint);

            if (operation.updateOne.collation)
                updateOperation.collation(operation.updateOne.collation);

            if (operation.updateOne.arrayFilters)
                updateOperation.arrayFilters(operation.updateOne.arrayFilters);

            updateOperation.updateOne(operation.updateOne.update);
        } else if (operation.updateMany) {
            if (!operation.updateMany.filter)
                nav_throwError('updateMany.filter for bulkWrite operation is required');

            if (!operation.updateMany.update)
                nav_throwError('updateMany.update for bulkWrite operation is required');

            var updateOperation = bulkOperation.find(operation.updateMany.filter);
            if (operation.updateMany.upsert)
                updateOperation = updateOperation.upsert();

            if (operation.updateMany.hint)
                updateOperation.hint(operation.updateMany.hint);

            if (operation.updateMany.collation)
                updateOperation.collation(operation.updateMany.collation);

            if (operation.updateMany.arrayFilters)
                updateOperation.arrayFilters(operation.updateMany.arrayFilters);

            updateOperation.update(operation.updateMany.update);
        } else if (operation.replaceOne) {
            if (!operation.replaceOne.filter)
                nav_throwError('replaceOne.filter for bulkWrite operation is required');

            if (!operation.replaceOne.replacement)
                nav_throwError('replaceOne.replacement for bulkWrite operation is required');

            var replaceOperation = bulkOperation.find(operation.replaceOne.filter);
            if (operation.replaceOne.upsert)
                replaceOperation = replaceOperation.upsert();

            if (operation.replaceOne.hint)
                replaceOperation.hint(operation.replaceOne.hint);

            if (operation.replaceOne.collation)
                replaceOperation.collation(operation.replaceOne.collation);

            replaceOperation.replaceOne(operation.replaceOne.replacement);
        } else if (operation.deleteOne) {
            if (!operation.deleteOne.filter)
                nav_throwError('deleteOne.filter for bulkWrite operation is required');

            var deleteOperation = bulkOperation.find(operation.deleteOne.filter);

            if (operation.deleteOne.collation)
                deleteOperation.collation(operation.deleteOne.collation);

            deleteOperation.removeOne();
        } else if (operation.deleteMany) {
            if (!operation.deleteMany.filter)
                nav_throwError('deleteMany.filter for bulkWrite operation is required');

            var deleteOperation = bulkOperation.find(operation.deleteMany.filter);

            if (operation.deleteMany.collation)
                deleteOperation.collation(operation.deleteMany.collation);

            deleteOperation.remove();
        }
    }, this);

    var reply = bulkOperation.execute(writeConcern);
    if (!result.acknowledged) {
        return result;
    }

    result.deletedCount = reply.nRemoved;
    result.insertedCount = reply.nInserted;
    result.matchedCount = reply.nMatched;
    result.modifiedCount = reply.nModified;
    result.upsertedCount = reply.nUpserted;
    result.insertedIds = insertedIds;
    result.upsertedIds = {};

    var upserts = reply.getUpsertedIds();
    upserts.forEach(function(x) {
        result.upsertedIds[x.index] = x._id;
    });

    return result;
};

NAVCollection.prototype.configureQueryAnalyzer = function (options) {
    if (!isObject(options))
        nav_throwError('argument for configureQueryAnalyzer should be an object');

    if (isUndefined(options.mode))
        nav_throwError('mode for configureQueryAnalyzer is required');
    else if (!isString(options.mode))
        nav_throwError('mode for configureQueryAnalyzer shoud be string');

    var fullName = this._db.getName() + "." + this.getName();
    return this._db.adminCommand({ configureQueryAnalyzer: fullName, ...options });
};

NAVCollection.prototype.convertToCapped = function (bytes) {
    if (!bytes)
        !throwError("convertToCapped should input # of bytes");
    var commandObj = { convertToCapped: this.getName(), size: bytes };
    return this.runCommand(commandObj);
};

NAVCollection.prototype.copyTo = function (newCollection) {
    return this._db.eval(function (collName, newCollection) {
        var from = db.getCollection(collName);
        var to = db.getCollection(newCollection);
        to.ensureIndex({ _id: 1 });
        var count = 0;

        var cursor = from.find();
        while (cursor.hasNext()) {
            var document = cursor.next();
            count++;
            to.save(document);
        }

        return count;
    }, this.getName(), newCollection);
};

NAVCollection.prototype.count = function (query, options) {
    var readConcern = undefined;
    var tmpQuery = Object.extend({}, (query || {}));
    var tmpOptions = Object.extend({}, (options || {}));
    if (tmpOptions && isObject(tmpOptions) && isNumber(tmpOptions.limit) && tmpOptions.limit == 0) {
        delete tmpOptions.limit;
    }
    if (tmpOptions && isObject(tmpOptions) && isString(tmpOptions.readConcern)) {
        readConcern = tmpOptions.readConcern;
        delete tmpOptions.readConcern;
    }
    if (!isObject(tmpQuery) || Object.keys(tmpQuery).length === 0)
        return this.forwardToCustomFunction("collectionEstimatedCount", tmpOptions, readConcern);
    return this.forwardToCustomFunction("collectionCount", tmpQuery, tmpOptions, readConcern);
};

NAVCollection.prototype.countDocuments = function (query, options) {
    var tmpQuery = Object.extend({}, (query || {}));
    var tmpOptions = Object.extend({}, (options || {}));
    if (tmpOptions && isObject(tmpOptions) && isNumber(tmpOptions.limit) && tmpOptions.limit == 0) {
        delete tmpOptions.limit;
    }
    return this.forwardToCustomFunction("collectionCount", tmpQuery, tmpOptions, undefined);
};

NAVCollection.prototype.createIndex = function (keys, options, commitQuorum) {
    if (arguments.length > 3)
        nav_throwError("invalid args");
    return this.createIndexes([keys], options, commitQuorum);
};

NAVCollection.prototype.createIndexes = function (keys, options, commitQuorum) {
    if (arguments.length > 3)
        nav_throwError("invalid args");

    // check keys isArray, options isObject
    if (!Array.isArray(keys))
        nav_throwError("invalid args");

    var commandObj = { createIndexes: this.getName() };

    var indexArray = Array(keys.length)
    for (var i = 0; i < keys.length; i++) {
        // new index name
        var indexName = "";
        for (var key in keys[i]) {
            var indexFieldType = keys[i][key];
            if (typeof indexFieldType == "function")
                continue;

            if (indexName.length > 0) indexName += "_";
            indexName += key + "_" + indexFieldType.toString();
        }
        // options
        var keyObj = { key: keys[i], name: indexName };
        if (options) {
            if (typeof (options) == "string")
                Object.extend(keyObj, { name: options });
            else if (typeof (options) == "boolean")
                Object.extend(keyObj, { unique: options });
            else if (typeof (options) == "object") {
                if (Array.isArray(options)) {
                    // index option only support three values: name, unique, dropDups
                    if (options.length > 3)
                        nav_throwError("invalid args");
                    var boolCount = 0;
                    for (var j = 0; j < options.length; j++) {
                        if (typeof (options[j]) == "string")
                            Object.extend(keyObj, { name: options[j] });
                        else if (typeof (options[j]) == "boolean") {
                            if (boolCount == 0)
                                Object.extend(keyObj, { unique: options[j] });
                            else
                                Object.extend(keyObj, { dropDups: options[j] });
                            boolCount++;
                        } else
                            nav_throwError("invalid args");
                    }
                } else {
                    Object.extend(keyObj, options);
                }
            } else {
                nav_throwError("invalid args");
            }
        }

        indexArray[i] = keyObj;
    }

    var indexObj = { indexes: indexArray };
    Object.extend(commandObj, indexObj);

    if (!(commitQuorum === undefined))
        Object.extend(commandObj, { commitQuorum: commitQuorum });

    return this.runCommand(commandObj);
};

NAVCollection.prototype.dataSize = function () {
    return this.stats().size;
};

// for deleteOne & deleteMany to call
NAVCollection.prototype.deleteOneOrMany = function (filter, options, deleteMany) {
    if (isUndefined(filter)) {
        if (deleteMany)
            nav_throwError("'filter' for collection.deleteMany is mssing");
        else
            nav_throwError("'filter' for collection.deleteOne is mssing");
    }

    options = Object.extend({}, options || {});

    // init result
    var result = { acknowledged: true };

    // parse input
    var writeConcern = this._extractWriteConcern(options);

    var collation = options.collation;
    var hint = options.hint;
    var letParams = options.let;
    if (writeConcern && isObject(writeConcern) && writeConcern.w == 0)
        result.acknowledged = false;

    options = {};
    if (writeConcern)
        options.writeConcern = writeConcern;
    if (collation)
        options.collation = collation;
    if (hint)
        options.hint = hint;
    if (letParams)
        options.let = letParams;

    var res = this.forwardToCustomFunction("collectionDeleteOneOrMany", filter, options, deleteMany);
    if (!result.acknowledged)
        return result;
    if (isObject(res)) {
        Object.extend(result, res);
        return result;
    }
}

NAVCollection.prototype.deleteOne = function (filter, options) {
    return this.deleteOneOrMany(filter, options, false);
};

NAVCollection.prototype.deleteMany = function (filter, options) {
    return this.deleteOneOrMany(filter, options, true);
};

NAVCollection.prototype.distinct = function (field, query, options) {
    var opts = Object.extend({}, options || {});

    if (!isString(field))
        nav_throwError("The first argument for distinct must be a string but was a " + typeof field);

    if (query != null && !isObject(query))
        nav_throwError("The query argument for distinct must be a document but was a " + typeof query);

    var command = { distinct: this.getName(), key: field, query: query || {} };

    if (opts.maxTimeMS)
        command.maxTimeMS = opts.maxTimeMS;

    if (opts.collation && this._db.getDatabaseVersion() >= 30400)
        command.collation = opts.collation;

    var result = this.runCommand(command);
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result.values;
};

NAVCollection.prototype.drop = function (options) {
    options = options || {};
    
    var command = Object.assign({drop: this.getName()}, options);

    return this.forwardToCustomFunction("collectionDrop", command);
};

NAVCollection.prototype.dropIndex = function (index) {
    return this.runCommand({ dropIndexes: this.getName(), index: index });
};

NAVCollection.prototype.dropIndexes = function (indexNames) {
    if (this._db.getDatabaseVersion() >= 40200) {
        indexNames = indexNames || '*';
        return this.runCommand({ dropIndexes: this.getName(), index: indexNames });
    } else {
        // no argument for dropIndexes
        if (arguments.length)
            nav_throwError("invalid args");
        return this.runCommand({ dropIndexes: this.getName(), index: "*" });
    }
};

NAVCollection.prototype.ensureIndex = function (keys, options) {
    return this.createIndex(keys, options);
};

NAVCollection.prototype.estimatedDocumentCount = function (options) {
    var command = { count: this.getName() };
    options = options || {};

    if (!isObject(options))
        nav_throwError("The options argument for estimatedDocumentCount must be a document but was a " + typeof options);

    if (options.maxTimeMS) {
        command.maxTimeMS = options.maxTimeMS;
    }

    var result = this.runCommand(command);

    if (!result.ok)
        nav_throwError(result.errmsg, result);
        
    return result.n;
};

NAVCollection.prototype.explain = function (verbosity) {
    if (verbosity && !isString(verbosity)) 
        verbosity = "allPlansExecution";
    if (!verbosity && !isString(verbosity))
        verbosity = "queryPlanner";
    if (verbosity !== "queryPlanner" && verbosity !== "executionStats" && verbosity !== "allPlansExecution")
        nav_throwError("explain verbosity must be one of {'queryPlanner', 'executionStats', 'allPlansExecution'}")

    return this.forwardToCustomFunction("collectionExplain", verbosity);
};

NAVCollection.prototype.find = function (query, projection, options) {
    if (this._db.getDatabaseVersion() >= 50000) {
        return this.forwardToCustomFunction("collectionFind", query, projection, options);
    } else {
        return this.forwardToCustomFunction("collectionFind", query, projection);
    }
};

NAVCollection.prototype.findAndModify = function (document) {
    var result = this.runCommand(Object.assign({
        findAndModify: this.getName()
    }, document));
    
    if (!result.ok)
        nav_throwError(result.errmsg, result);
    return result.value; 
};

NAVCollection.prototype.findOne = function (query, projection, options) {
    var cursor = undefined;
    if (this._db.getDatabaseVersion() >= 50000 && !isUndefined(options)) {
        cursor = this.find(query, projection, options).limit(1);
    } else {
        cursor = this.find(query, projection).limit(1);
    }

    if (!cursor.hasNext())
        return null;
    return cursor.next();
};

NAVCollection.prototype.findOneAndDelete = function (filter, options) {
    var document = {
        query: filter,
        remove: true
    };

    if (typeof options != "object")
        return this.findAndModify(document);

    var propMap = new Map([
        ["projection", "fields"],
        ["sort", "sort"],
        ["maxTimeMS", "maxTimeMS"]
    ]);

    if (options && options.collation && this._db.getDatabaseVersion() >= 30400)
        propMap.set("collation", "collation");

    propMap.forEach(function(docProp, optProp){
        if (optProp in options)
            document[docProp] = options[optProp]; 
    });
    
    var writeConcern = this._extractWriteConcern(options);
    
    if (writeConcern) {
        document.writeConcern = writeConcern;
    }

    return this.findAndModify(document);
};

NAVCollection.prototype.findOneAndReplace = function (filter, replacement, options) {
    if (typeof replacement != "object")
        nav_throwError("invalid args");

    if (Array.isArray(replacement))
        nav_throwError('collection.findOneAndReplace should not contains pipeline-style update"');
    
    var props = Object.keys(replacement);
    if (props.length > 0 && props[0].startsWith("$"))
        nav_throwError("invalid args");

    var document = {
        query: filter,
        update: replacement
    };

    if (typeof options != "object")
        return this.findAndModify(document);

    if (isString(options.returnDocument)) {
        if (options.returnDocument == 'before') {
            options.returnNewDocument = false;
        } else if (options.returnDocument == 'after') {
            options.returnNewDocument = true;
        }
    }

    var propMap = new Map([
        ["projection", "fields"],
        ["sort", "sort"],
        ["maxTimeMS", "maxTimeMS"],
        ["upsert", "upsert"],
        ["returnNewDocument", "new"],
    ]);

    if (options && options.collation && this._db.getDatabaseVersion() >= 30400)
        propMap.set("collation", "collation");

    if (options && options.hint && this._db.getDatabaseVersion() >= 40201)
        propMap.set("hint", "hint");

    propMap.forEach(function(docProp, optProp){
        if (optProp in options)
            document[docProp] = options[optProp]; 
    });
    
    var writeConcern = this._extractWriteConcern(options);

    if (writeConcern) {
        document.writeConcern = writeConcern;
    }

    return this.findAndModify(document);
};

NAVCollection.prototype.findOneAndUpdate = function (filter, update, options) {
    if (!Array.isArray(update)) {
        if (typeof update != "object")
            nav_throwError("invalid args");

        var props = Object.keys(update);
        if (props.length == 0 || !props[0].startsWith("$"))
            nav_throwError("invalid args");
    }

    var document = {
        query: filter,
        update: update
    };

    if (typeof options != "object")
        return this.findAndModify(document);
        
    if (isString(options.returnDocument)) {
        if (options.returnDocument == 'before') {
            options.returnNewDocument = false;
        } else if (options.returnDocument == 'after') {
            options.returnNewDocument = true;
        }
    }

    var propMap = new Map([
        ["projection", "fields"],
        ["sort", "sort"],
        ["maxTimeMS", "maxTimeMS"],
        ["upsert", "upsert"],
        ["returnNewDocument", "new"]
    ]);

    if (options && options.collation && this._db.getDatabaseVersion() >= 30400)
        propMap.set("collation", "collation");

    if (options && options.arrayFilters && this._db.getDatabaseVersion() >= 30600)
        propMap.set("arrayFilters", "arrayFilters");

    if (options && options.hint && this._db.getDatabaseVersion() >= 40201)
        propMap.set("hint", "hint");
 
    propMap.forEach(function(docProp, optProp){
        if (optProp in options)
            document[docProp] = options[optProp]; 
    });
    
    var writeConcern = this._extractWriteConcern(options);

    if (writeConcern) {
        document.writeConcern = writeConcern;
    }

    return this.findAndModify(document);
};

NAVCollection.prototype.getDB = function () {
    return this._db;
};

NAVCollection.prototype.getPlanCache = function () {
    return this.forwardToCustomFunction("collectionGetQueryPlanCache");
};

NAVCollection.prototype.getIndexes = function () {
    return this.forwardToCustomFunction("collectionGetIndexes").toArray();
};

NAVCollection.prototype.getName = function () {
    return this.forwardToCustomFunction("collectionGetName");
};

NAVCollection.prototype._isSharded = function () {
    var query = {};
    query._id = this._db.getName() + "." + this.getName();
    query.dropped = { $ne: true };

    return !!this._db.getSiblingDB("config").collections.countDocuments(query);
};

NAVCollection.prototype.getShardDistribution = function () {

    var configDB = this._db.getSiblingDB("config");

    if (!this._isSharded()) {
        print("Collection " + this.getName() + " is not sharded.");
        return;
    }

    var dataSizeFormat = function (bytes) {
        if (bytes == null)
            return "0B";
        if (bytes < 1024)
            return Math.floor(bytes) + "B";
        if (bytes < 1024 * 1024)
            return Math.floor(bytes / 1024) + "KiB";
        if (bytes < 1024 * 1024 * 1024)
            return Math.floor((Math.floor(bytes / 1024) / 1024) * 100) / 100 + "MiB";
        return Math.floor((Math.floor(bytes / (1024 * 1024)) / 1024) * 100) / 100 + "GiB";
    };

    var allStats = this.aggregate({ "$collStats": { storageStats: {} } });

    var totals = { numChunks: 0, size: 0, count: 0 };
    var conciseShardsStats = [];

    allStats.forEach(function (stats) {
        var newVersion = configDB.collections.countDocuments(
            { _id: stats.ns, timestamp: { $exists: true } }, { limit: 1 });
        var collUuid = configDB.collections.findOne({ _id: stats.ns }).uuid;
        var shardStats = {
            shardId: stats.shard,
            host: configDB.shards.findOne({ _id: stats.shard }).host,
            size: stats.storageStats.size,
            count: stats.storageStats.count,
            numChunks: (newVersion ? configDB.chunks.countDocuments(
                { uuid: collUuid, shard: stats.shard })
                : configDB.chunks.countDocuments(
                    { ns: stats.ns, shard: stats.shard })),
            avgObjSize: stats.storageStats.avgObjSize
        };

        print("\nShard " + shardStats.shardId + " at " + shardStats.host);

        var estChunkData =
            (shardStats.numChunks == 0) ? 0 : (shardStats.size / shardStats.numChunks);
        var estChunkCount =
            (shardStats.numChunks == 0) ? 0 : Math.floor(shardStats.count / shardStats.numChunks);
        print(" data : " + dataSizeFormat(shardStats.size) + " docs : " + shardStats.count +
            " chunks : " + shardStats.numChunks);
        print(" estimated data per chunk : " + dataSizeFormat(estChunkData));
        print(" estimated docs per chunk : " + estChunkCount);

        totals.size += shardStats.size;
        totals.count += shardStats.count;
        totals.numChunks += shardStats.numChunks;

        conciseShardsStats.push(shardStats);
    });

    print("\nTotals");
    print(" data : " + dataSizeFormat(totals.size) + " docs : " + totals.count + " chunks : " + totals.numChunks);
    for (let stats of conciseShardsStats) {
        var estDataPercent = Math.floor(stats.size / totals.size * 10000) / 100;
        var estDocPercent = Math.floor(stats.count / totals.count * 10000) / 100;

        print(" Shard " + stats.shardId + " contains " +
              estDataPercent + "% data, " +
              estDocPercent + "% docs in cluster, " +
            "avg obj size on shard : " + dataSizeFormat(stats.avgObjSize));
    }

    print("\n");
};

NAVCollection.prototype.getShardVersion = function () {
    var fullName = this._db.getName() + "." + this.getName();
    return this._db.adminCommand({ getShardVersion: fullName });
};

NAVCollection.prototype.group = function(document) {
    var options = {
        ns: this.getName()
    };

    var propMap = new Map([
        ["key", "key"],
        ["reduce", "$reduce"],
        ["initial", "initial"],
        ["keyf", "$keyf"],
        ["cond", "cond"],
        ["finalize", "finalize"],
        ["collation", "collation"]
    ]);

    propMap.forEach(function(optProp, docProp){
        if (docProp in document)
            options[optProp] = document[docProp]; 
    });

    var result = this.runCommand({
        group: options
    });

    if (!result.ok)
        nav_throwError(result.errmsg, result);
    
    return result.retval;
};

NAVCollection.prototype.help = function () {
    var collectionName = this.getName();
    print("DBCollection help\n" +
        "\tdb." + collectionName + ".analyzeShardKey(key, <opts>) - returns metrics for evaluating a shard key\n" +
        "\tdb." + collectionName + ".find().help() - show DBCursor help\n" +
        "\tdb." + collectionName + ".bulkWrite( operations, <optional params> ) - bulk execute write operations, optional parameters are: w, wtimeout, j\n" +
        "\tdb." + collectionName + ".configureQueryAnalyzer(options) - starts or stops collecting metrics about reads and writes against an unsharded or sharded collection\n" +
        "\tdb." + collectionName + ".count( query = {}, <optional params> ) - count the number of documents that matches the query, optional parameters are: limit, skip, hint, maxTimeMS\n" +
        "\tdb." + collectionName + ".countDocuments( query = {}, <optional params> ) - count the number of documents that matches the query, optional parameters are: limit, skip, hint, maxTimeMS\n" +
        "\tdb." + collectionName + ".copyTo(newColl) - duplicates collection by copying all documents to newColl; no indexes are copied.\n" +
        "\tdb." + collectionName + ".convertToCapped(maxBytes) - calls {convertToCapped:'" + collectionName + "', size:maxBytes}} command\n" +
        "\tdb." + collectionName + ".createIndex(keypattern[,options])\n" +
        "\tdb." + collectionName + ".createIndexes([keypatterns], <options>)\n" +
        "\tdb." + collectionName + ".dataSize()\n" +
        "\tdb." + collectionName + ".deleteOne( filter, <optional params> ) - delete first matching document, optional parameters are: w, wtimeout, j\n" +
        "\tdb." + collectionName + ".deleteMany( filter, <optional params> ) - delete all matching documents, optional parameters are: w, wtimeout, j\n" +
        "\tdb." + collectionName + ".distinct( key, query, <optional params> ) - e.g. db." + collectionName + ".distinct( 'x' ), optional parameters are: maxTimeMS\n" +
        "\tdb." + collectionName + ".drop() drop the collection\n" +
        "\tdb." + collectionName + ".dropIndex(index) - e.g. db." + collectionName + ".dropIndex( \"indexName\" ) or db." + collectionName + ".dropIndex( { \"indexKey\" : 1 } )\n" +
        "\tdb." + collectionName + ".dropIndexes()\n" +
        "\tdb." + collectionName + ".ensureIndex(keypattern[,options]) - DEPRECATED, use createIndex() instead\n" +
        "\tdb." + collectionName + ".estimatedDocumentCount( <optional params> ) - estimate the document count using collection metadata, optional parameters are: maxTimeMS\n" +
        "\tdb." + collectionName + ".explain().help() - show explain help\n" +
        "\tdb." + collectionName + ".reIndex()\n" +
        "\tdb." + collectionName + ".find([query],[fields]) - query is an optional query filter. fields is optional set of fields to return.\n" +
        "\t                                              e.g. db." + collectionName + ".find({ x:77 }, { name:1, x : 1 })\n" +
        "\tdb." + collectionName + ".find(...).count()\n" +
        "\tdb." + collectionName + ".find(...).limit(n)\n" +
        "\tdb." + collectionName + ".find(...).skip(n)\n" +
        "\tdb." + collectionName + ".find(...).sort(...)\n" +
        "\tdb." + collectionName + ".findOne([query], [fields], [options], [readConcern])\n" +
        "\tdb." + collectionName + ".findOneAndDelete( filter, <optional params> ) - delete first matching document, optional parameters are: projection, sort, maxTimeMS\n" +
        "\tdb." + collectionName + ".findOneAndReplace( filter, replacement, <optional params> ) - replace first matching document, optional parameters are: projection, sort, maxTimeMS, upsert, returnNewDocument\n" +
        "\tdb." + collectionName + ".findOneAndUpdate( filter, <update object or pipeline>, <optional params> ) - update first matching document, optional parameters are: projection, sort, maxTimeMS, upsert, returnNewDocument\n" +
        "\tdb." + collectionName + ".getDB() get DB object associated with collection\n" +
        "\tdb." + collectionName + ".getPlanCache() get query plan cache associated with collection\n" +
        "\tdb." + collectionName + ".getIndexes()\n" +
        "\tdb." + collectionName + ".group( { key : ..., initial: ..., reduce : ...[, cond: ...] } )\n" +
        "\tdb." + collectionName + ".hideIndex(index) - e.g. db." + collectionName + ".hideIndex( \"indexName\" ) or db." + collectionName + ".hideIndex( { \"indexKey\" : 1 } )\n" +
        "\tdb." + collectionName + ".unhideIndex(index) - e.g. db." + collectionName + ".unhideIndex( \"indexName\" ) or db." + shortName + ".unhideIndex( { \"indexKey\" : 1 } )\n" +
        "\tdb." + collectionName + ".insert(obj)\n" +
        "\tdb." + collectionName + ".insertOne( obj, <optional params> ) - insert a document, optional parameters are: w, wtimeout, j\n" +
        "\tdb." + collectionName + ".insertMany( [objects], <optional params> ) - insert multiple documents, optional parameters are: w, wtimeout, j\n" +
        "\tdb." + collectionName + ".mapReduce( mapFunction , reduceFunction , <optional params> )\n" +
        "\tdb." + collectionName + ".aggregate( [pipeline], <optional params> ) - performs an aggregation on a collection; returns a cursor\n" +
        "\tdb." + collectionName + ".remove(query)\n" +
        "\tdb." + collectionName + ".replaceOne( filter, replacement, <optional params> ) - replace the first matching document, optional parameters are: upsert, w, wtimeout, j\n" +
        "\tdb." + collectionName + ".renameCollection( newName , <dropTarget> ) renames the collection.\n" +
        "\tdb." + collectionName + ".runCommand( name , <options> ) runs a db command with the given name where the first param is the collection name\n" +
        "\tdb." + collectionName + ".save(obj)\n" +
        "\tdb." + collectionName + ".stats({scale: N, indexDetails: true/false, " + "indexDetailsKey: <index key>, indexDetailsName: <index name>})\n" +
        "\tdb." + collectionName + ".storageSize() - includes free space allocated to this collection\n" +
        "\tdb." + collectionName + ".totalIndexSize() - size in bytes of all the indexes\n" +
        "\tdb." + collectionName + ".totalSize() - storage allocated for all data and indexes\n" +
        "\tdb." + collectionName + ".update( query, <update object or pipeline>[, upsert_bool, multi_bool] ) - instead of two flags, you can pass an object with fields: upsert, multi, hint\n" +
        "\tdb." + collectionName + ".updateOne( filter, <update object or pipeline>, <optional params> ) - update the first matching document, optional parameters are: upsert, w, wtimeout, j, hint\n" +
        "\tdb." + collectionName + ".updateMany( filter, <update object or pipeline>, <optional params> ) - update all matching documents, optional parameters are: upsert, w, wtimeout, j, hint\n" +
        "\tdb." + collectionName + ".validate( <full> ) - SLOW\n" +
        "\tdb." + collectionName + ".getShardVersion() - only for use with sharding\n" +
        "\tdb." + collectionName + ".getShardDistribution() - prints statistics about data distribution in the cluster\n" +
        "\tdb." + collectionName + ".latencyStats() - display operation latency histograms for this collection\n");
};

NAVCollection.prototype._hiddenIndex = function (index, hidden) {
    var indexField = {};
    if (isString(index)) {
        indexField = { name: index, hidden: hidden };
    } else if (isObject(index)) {
        indexField = { keyPattern: index, hidden: hidden };
    } else {
        nav_throwError("Index must be either the index name or the index specification document");
    }
    var command = { "collMod": this.getName(), index: indexField };
    var result = this.runCommand(command);
    return result;
};

NAVCollection.prototype.hideIndex = function (index) {
    return this._hiddenIndex(index, true);
};

NAVCollection.prototype.initializeOrderedBulkOp = function () {
    return this.forwardToCustomFunction("collectionInitializeOrderedBulkOp");
};

NAVCollection.prototype.initializeUnorderedBulkOp = function () {
    return this.forwardToCustomFunction("collectionInitializeUnorderedBulkOp");
};

NAVCollection.prototype.insert = function (document, options) {
    // input validation
    if (isUndefined(document) || !isObject(document))
        nav_throwError("collection.insert requires document or list of documents");
        
    return this._insertOneOrMany(document, options, Array.isArray(document));
};

NAVCollection.prototype._insertOneOrMany = function (document, options, insertMany) {
    if (isUndefined(document)) {
        if (insertMany)
            nav_throwError("'documents' for collection.insertMany is mssing");
        else
            nav_throwError("'document' for collection.insertOne is mssing");
    }
    
    if (insertMany && !Array.isArray(document))
        nav_throwError("collection.insertMany requires list of documents");
    else if (!insertMany && Array.isArray(document))
        nav_throwError("collection.insertOne requires a single document");

    options = Object.extend({}, options || {});

    // init result
    var result = { acknowledged: true };

    // parse input
    var writeConcern = this._extractWriteConcern(options);

    var ordered = options.ordered;
    if (writeConcern && isObject(writeConcern) && writeConcern.w == 0)
        result.acknowledged = false;

    options = {};
    if (writeConcern)
        options.writeConcern = writeConcern;
    if (isBoolean(ordered))
        options.ordered = ordered;
        
    if (Array.isArray(document)) {
        var newDocuments = [];
        for (var i = 0; i < document.length; i++) {
            var documentNew = document[i];
            if (isObject(documentNew)) {
                if (isUndefined(documentNew._id)) {
                    var documentOld = documentNew;
                    documentNew = { _id: new ObjectId() };
                    Object.extend(documentNew, documentOld);
                }
            }
            newDocuments.push(documentNew);
        }
        document = newDocuments;
    } else if (isObject(document)) {
        if (isUndefined(document._id)) {
            var documentOld = document;
            document = { _id: new ObjectId() };
            Object.extend(document, documentOld);
        }
    }

    var res = this.forwardToCustomFunction("collectionInsertOneOrMany", document, options, insertMany);
    if (!result.acknowledged)
        return result;
    if (isObject(res)) {
        Object.extend(result, res);
        return result;
    }
}

NAVCollection.prototype.insertOne = function (document, options) {
    return this._insertOneOrMany(document, options, false);
};

NAVCollection.prototype.insertMany = function (documents, options) {
    return this._insertOneOrMany(documents, options, true);
};

NAVCollection.prototype.isCapped = function () {
    var stats = this.stats();
    if (stats.capped)
        return true;
    return false;
};

NAVCollection.prototype.latencyStats = function (options) {
    options = options || {};

    return this.aggregate([{
        $collStats: {
            latencyStats: options
        }
    }]);
};

NAVCollection.prototype.mapReduce = function (map, reduce, options) {
    var result = this.runCommand(Object.assign({
        mapReduce: this.getName(),
        map: map,
        reduce: reduce
    }, options));

    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result;
};

NAVCollection.prototype.reIndex = function () {
    // no api for reIndex
    return this.runCommand({ reIndex: this.getName() });
};

NAVCollection.prototype.remove = function (query, justOne) {
    // parse input
    var parsed = _parseRemove(query, justOne);
    query = parsed.query;
    justOne = parsed.justOne;
    var writeConcern = parsed.wc;
    var collation = parsed.collation;
    var letParams = parsed.let;

    var options = {};
    if (writeConcern)
        options.writeConcern = writeConcern;
    if (collation)
        options.collation = collation;
    if (letParams)
        options.let = letParams;

    return this.deleteOneOrMany(query, options, !justOne);        
};

NAVCollection.prototype.renameCollection = function (target, dropTarget) {
    if (arguments.length === 1 && isObject(target)) {
        if (target.hasOwnProperty('dropTarget')) {
            dropTarget = target['dropTarget'];
        }
        target = target['to'];
    }

    if (typeof dropTarget == "undefined")
        dropTarget = false;
        
    if (typeof target != "string" || typeof dropTarget != "boolean")
        nav_throwError("invalid args");

    return this._db.adminCommand({
        renameCollection: this._db.getName() + "." + this.getName(),
        to: this._db.getName() + "." + target,
        dropTarget: dropTarget 
    });
};

NAVCollection.prototype.replaceOne = function (filter, replacement, options) {
    if (isUndefined(filter))
        nav_throwError("filter for collection.collection.replaceOne is mssing");

    // input validation
    if (Array.isArray(replacement))
        nav_throwError("collection.replaceOne should not contains pipeline-style update");

    // input validation
    var keys = Object.keys(replacement);
    if (keys.length > 0 && keys[0].length > 0 && keys[0][0] == "$")
        nav_throwError("collection.replaceOne should not contains any update operator");

    options = Object.extend({}, options || {});

    // init result
    var result = { acknowledged: true };

    // parse input
    var writeConcern = this._extractWriteConcern(options);

    var upsert = options.upsert;
    var collation = options.collation;
    var hint = options.hint;
    if (writeConcern && isObject(writeConcern) && writeConcern.w == 0)
        result.acknowledged = false;

    options = {};
    if (writeConcern)
        options.writeConcern = writeConcern;
    if (upsert)
        options.upsert = upsert;
    if (collation)
        options.collation = collation;
    if (hint)
        options.hint = hint;

    var res = this.forwardToCustomFunction("collectionReplaceOne", filter, replacement, options);
    if (!result.acknowledged)
        return result;
    if (isObject(res)) {
        Object.extend(result, res);
        return result;
    }
};

NAVCollection.prototype.runCommand = function (document) {
    return this._db.runCommand(document);
};

NAVCollection.prototype.save = function (document, options) {
    if (!document)
        nav_throwError("db.collection.save requires document object");

    if (isNumber(document) || isString(document))
        nav_throwError("db.collection.save only accept document object");

    if (isUndefined(document._id)) {
        document._id = new ObjectId();
        return this.insert(document, options);
    } else {
        options = options || {};
        options.upsert = true;
        return this.update({ _id: document._id }, document, options);
    }
};

NAVCollection.prototype.stats = function (originalOptions) {
    var options = {};
    if (isNumber(originalOptions))
        options.scale = originalOptions;
    else if (isObject(originalOptions))
        options = originalOptions;

    if (options.indexDetailsKey && options.indexDetailsName)
        nav_throwError("Cannot filter indexDetails on indexDetailsKey and indexDetailsName at the same time");

    if (options.indexDetailsKey && !isObject(options.indexDetailsKey))
        nav_throwError("Expected options.indexDetailsKey to be a document, but was " + typeof options.indexDetailsKey);

    if (options.indexDetailsName && !isString(options.indexDetailsName))
        nav_throwError("Expected options.indexDetailsName to be a string, but was " + typeof options.indexDetailsName);

    options.scale = options.scale || 1;
    options.indexDetails = options.indexDetails || false;

    var _getAggregatedCollStats = function (scale, db, collection) {
        var _scaleIndividualShardStatistics = function (shardStats, scale) {
            var scaledStats = {};

            for (const fieldName of Object.keys(shardStats)) {
                if (['size', 'maxSize', 'storageSize', 'totalIndexSize', 'totalSize'].includes(fieldName)) {
                    scaledStats[fieldName] = Number(shardStats[fieldName]) / scale;
                } else if (fieldName === 'scaleFactor') {
                    scaledStats[fieldName] = scale;
                } else if (fieldName === 'indexSizes') {
                    const scaledIndexSizes = {};
                    for (const indexKey of Object.keys(shardStats[fieldName])) {
                        scaledIndexSizes[indexKey] = Number(shardStats[fieldName][indexKey]) / scale;
                    }
                    scaledStats[fieldName] = scaledIndexSizes;
                } else {
                    scaledStats[fieldName] = shardStats[fieldName];
                }
            }

            return scaledStats;
        }

        var _aggregateAndScaleCollStats = function (collStats, scale) {
            var result = {};

            var shardStats = {};
            var counts = {};
            var indexSizes = {};
            var clusterTimeseriesStats = {};

            var maxSize = 0;
            var unscaledCollSize = 0;

            var nindexes = 0;
            var timeseriesBucketsNs;
            var timeseriesTotalBucketSize = 0;

            for (const shardResult of collStats) {
                const shardStorageStats = shardResult.storageStats;

                const countField = shardStorageStats.count;
                const shardObjCount = !isUndefined(countField) ? countField : 0;

                for (const fieldName of Object.keys(shardStorageStats)) {
                    if (['ns', 'ok', 'lastExtentSize', 'paddingFactor'].includes(fieldName)) {
                        continue;
                    }
                    if (['userFlags', 'capped', 'max', 'paddingFactorNote', 'indexDetails', 'wiredTiger'].includes(fieldName)) {
                        result[fieldName] ??= shardStorageStats[fieldName];
                    } else if (fieldName === 'timeseries') {
                        const shardTimeseriesStats = shardStorageStats[fieldName];
                        for (const [timeseriesStatName, timeseriesStat] of Object.entries(shardTimeseriesStats)) {
                            if (isString(timeseriesStat)) {
                                if (!timeseriesBucketsNs) {
                                    timeseriesBucketsNs = timeseriesStat;
                                }
                            } else if (timeseriesStatName === 'avgBucketSize') {
                                timeseriesTotalBucketSize += Number(shardTimeseriesStats.bucketCount) * Number(timeseriesStat);
                            } else {
                                if (isUndefined(clusterTimeseriesStats[timeseriesStatName])) {
                                    clusterTimeseriesStats[timeseriesStatName] = 0;
                                }
                                clusterTimeseriesStats[timeseriesStatName] += Number(timeseriesStat);
                            }
                        }
                    } else if (['count', 'size', 'storageSize', 'totalIndexSize', 'totalSize', 'numOrphanDocs'].includes(fieldName)) {
                        if (isUndefined(counts[fieldName])) {
                            counts[fieldName] = 0;
                        }
                        counts[fieldName] += Number(shardStorageStats[fieldName]);
                    } else if (fieldName === 'avgObjSize') {
                        const shardAvgObjSize = Number(shardStorageStats[fieldName]);
                        unscaledCollSize += shardAvgObjSize * shardObjCount;
                    } else if (fieldName === 'maxSize') {
                        const shardMaxSize = Number(shardStorageStats[fieldName]);
                        maxSize = Math.max(maxSize, shardMaxSize);
                    } else if (fieldName === 'indexSizes') {
                        for (const indexName of Object.keys(shardStorageStats[fieldName])) {
                            if (isUndefined(indexSizes[indexName])) {
                                indexSizes[indexName] = 0;
                            }
                            indexSizes[indexName] += Number(shardStorageStats[fieldName][indexName]);
                        }
                    } else if (fieldName === 'nindexes') {
                        const shardIndexes = shardStorageStats[fieldName];

                        if (nindexes === 0) {
                            nindexes = shardIndexes;
                        } else if (shardIndexes > nindexes) {
                            nindexes = shardIndexes;
                        }
                    }
                }

                if (shardResult.shard) {
                    shardStats[shardResult.shard] = _scaleIndividualShardStatistics(shardStorageStats, scale);
                }
            }

            var ns = db.getName() + "." + collection.getName();
            const config = db.getSiblingDB('config');
            if (collStats[0].shard) {
                result.shards = shardStats;
            }

            try {
                var filter = {};
                filter._id = timeseriesBucketsNs ?? ns;
                filter.dropped = { $ne: true };
                filter.unsplittable = { $ne: true }
                result.sharded = !!(config.getCollection('collections').findOne(filter));
            } catch (err) {
                result.sharded = collStats.length > 1;
            }

            for (const [countField, count] of Object.entries(counts)) {
                if (['size', 'storageSize', 'totalIndexSize', 'totalSize'].includes(countField)) {
                    result[countField] = count / scale;
                } else {
                    result[countField] = count;
                }
            }
            if (timeseriesBucketsNs && Object.keys(clusterTimeseriesStats).length > 0) {
                result.timeseries = {
                    ...clusterTimeseriesStats,
                    avgBucketSize: clusterTimeseriesStats.bucketCount ? timeseriesTotalBucketSize / clusterTimeseriesStats.bucketCount : 0,
                    bucketsNs: timeseriesBucketsNs,
                };
            }
            result.indexSizes = {};
            for (const [indexName, indexSize] of Object.entries(indexSizes)) {
                result.indexSizes[indexName] = indexSize / scale;
            }
            if (counts.count > 0) {
                result.avgObjSize = unscaledCollSize / counts.count;
            } else {
                result.avgObjSize = 0;
            }
            if (result.capped) {
                result.maxSize = maxSize / scale;
            }
            result.ns = ns;
            result.nindexes = nindexes;
            if (!isUndefined(collStats[0].storageStats.scaleFactor)) {
                result.scaleFactor = scale;
            }
            result.ok = 1;

            return result;
        }

        var fullName = db.getName() + "." + collection.getName();
        var version = db.getDatabaseVersion();
        if (version >= 36000) {
            var pipeline = [{ $collStats: { storageStats: { scale: 1, } } }];
            var collStats = collection.aggregate(pipeline).toArray();

            if (!collStats || isUndefined(collStats[0]))
                nav_throwError("Error running $collStats aggregation stage on " + fullName);

            return _aggregateAndScaleCollStats(collStats, scale);
        } else {
            var command = {
                collStats: collection.getName(),
                scale: scale || 1,
            };
            var result = collection.forwardToCustomFunction("collectionReadCommand", command);

            if (!result) {
                nav_throwError("Error running collStats command on " + fullName);
            }

            return result;
        }
    }


    var result = _getAggregatedCollStats(options.scale, this._db, this);

    var _getIndexName = function (collection, indexKey) {
        if (!isObject(indexKey))
            return undefined;
        var indexName;
        collection.getIndexes().forEach(function (spec) {
            if (friendlyEqual(spec.key, indexKey)) {
                indexName = spec.name;
            }
        });
        return indexName;
    };

    var filterIndexName = options.indexDetailsName || _getIndexName(this, options.indexDetailsKey);

    var _updateStats = function (stats) {
        if (!isObject(stats))
            return;
        if (!stats.indexDetails)
            return;
        if (!options.indexDetails) {
            delete stats.indexDetails;
            return;
        }
        if (!filterIndexName)
            return;
        for (const key of Object.keys(stats.indexDetails)) {
            if (key === filterIndexName) {
                continue;
            }
            delete stats.indexDetails[key];
        }
    };
    _updateStats(result);

    for (const shardName of Object.keys(result.shards ?? {})) {
        _updateStats(result.shards[shardName]);
    }
    return result;
};

NAVCollection.prototype.storageSize = function () {
    return this.stats().storageSize;
};

NAVCollection.prototype.totalIndexSize = function () {
    return this.stats().totalIndexSize;
};

NAVCollection.prototype.totalSize = function () {
    var stats = this.stats();
    var totalSize = stats.storageSize;
    var totalIndexSize = stats.totalIndexSize;
    if (totalIndexSize)
        totalSize += totalIndexSize;

    return totalSize;
};
NAVCollection.prototype.unhideIndex = function (index) {
    return this._hiddenIndex(index, false);
};

NAVCollection.prototype.update = function (query, updateSpec, upsert, multi) {
    if (isUndefined(query))
        nav_throwError("'query' for collection.update is mssing");

    // parse input
    var parsed = _parseUpdate(query, updateSpec, upsert, multi);
    var query = parsed.query;
    var updateSpec = parsed.updateSpec;
    var hint = parsed.hint;
    var upsert = parsed.upsert;
    var multi = parsed.multi;
    var writeConcern = parsed.wc;
    var collation = parsed.collation;
    var arrayFilters = parsed.arrayFilters;
    var letParams = parsed.let;
    
    if (!writeConcern)
        writeConcern = this._getWriteConcern();
    
    // init result
    var result = { acknowledged: true };
    
    if (writeConcern && isObject(writeConcern) && writeConcern.w == 0)
        result.acknowledged = false;

    var options = {};
    if (upsert)
        options.upsert = upsert;
    if (writeConcern)
        options.writeConcern = writeConcern;
    if (hint)
        options.hint = hint;
    if (collation)
        options.collation = collation;
    if (arrayFilters)
        options.arrayFilters = arrayFilters;
    if (letParams)
        options.let = letParams;

    return this.updateOneOrMany(query, updateSpec, options, multi);
};

NAVCollection.prototype.updateOneOrMany = function (query, update, options, updateMany) {
    if (isUndefined(query)) {
        if (updateMany)
            nav_throwError("'query' for collection.updateMany is mssing");
        else
            nav_throwError("'query' for collection.updateOne is mssing");
    }

    // input validation
    if (!Array.isArray(update)) {
        var keys = Object.keys(update);
        var functionName = "collection.";
        if (updateMany)
            functionName += "updateMany";
        else
            functionName += "updateOne";
        if (keys.length = 0)
            nav_throwError(functionName + " requires at least one update operator");
        if (keys.length > 0 && keys[0].length > 0 && keys[0][0] != "$")
            nav_throwError(functionName + " requires update operator");
    }

    options = Object.extend({}, options || {});

    // init result
    var result = { acknowledged: true };

    // parse input
    var writeConcern = this._extractWriteConcern(options);

    var upsert = options.upsert;
    var hint = options.hint;
    var collation = options.collation;
    var arrayFilters = options.arrayFilters;
    var letParams = options.let;
    if (writeConcern && isObject(writeConcern) && writeConcern.w == 0)
        result.acknowledged = false;

    options = {};
    if (upsert)
        options.upsert = upsert;
    if (hint)
        options.hint = hint;
    if (collation)
        options.collation = collation;
    if (arrayFilters)
        options.arrayFilters = arrayFilters;
    if (letParams)
        options.let = letParams;

    var res = this.forwardToCustomFunction("collectionUpdateOneOrMany", query, update, options, updateMany);
    if (!result.acknowledged)
        return result;
    if (isObject(res)) {
        Object.extend(result, res);
        return result;
    }
}

NAVCollection.prototype.updateOne = function (query, update, options) {
    return this.updateOneOrMany(query, update, options, false);
};

NAVCollection.prototype.updateMany = function (query, update, options) {
    return this.updateOneOrMany(query, update, options, true);
};

NAVCollection.prototype.validate = function (options) {
    if (!isObject(options) && !isUndefined(options) && !isBoolean(options))
        return "options should be document";

    var command = { validate: this.getName() };
    if (isObject(options)) {
        Object.assign(command, options || {});
    } else if (isBoolean(options)) {
        command.full = options ? true : false;
    }

    var result = this.forwardToCustomFunction("collectionValidate", command);

    if (isUndefined(result.valid)) {
        result.valid = false;

        var raw = result.result || result.raw;

        if (raw) {
            var str = "-" + tojson(raw);
            result.valid = !(str.match(/exception/) || str.match(/corrupt/));

            var p = /lastExtentSize:(\d+)/;
            var r = p.exec(str);
            if (r) {
                result.lastExtentSize = Number(r[1]);
            }
        }
    }

    return result;
};

NAVCollection.prototype.watch = function (pipeline, options) {
    var tmpPipeline = Object.extend([], (pipeline || []));
    var tmpOptions = Object.extend({}, (options || {}));
    if (!tmpPipeline instanceof Array)
        nav_throwError("pipeline should be array");
    if (!tmpOptions instanceof Object)
        nav_throwError("options should be object");

    var changeStreamStage = { fullDocument: tmpOptions.fullDocument || "default" };
    delete tmpOptions.fullDocument;
    
    if (tmpOptions.hasOwnProperty("resumeAfter")) {
        changeStreamStage.resumeAfter = tmpOptions.resumeAfter;
        delete tmpOptions.resumeAfter;
    }

    if (tmpOptions.hasOwnProperty("startAfter")) {
        changeStreamStage.startAfter = tmpOptions.startAfter;
        delete tmpOptions.startAfter;
    }

    if (tmpOptions.hasOwnProperty("fullDocumentBeforeChange")) {
        changeStreamStage.fullDocumentBeforeChange = tmpOptions.fullDocumentBeforeChange;
        delete tmpOptions.fullDocumentBeforeChange;
    }

    if (tmpOptions.hasOwnProperty("allChangesForCluster")) {
        changeStreamStage.allChangesForCluster = tmpOptions.allChangesForCluster;
        delete tmpOptions.allChangesForCluster;
    }

    if (tmpOptions.hasOwnProperty("allowToRunOnConfigDB")) {
        changeStreamStage.allowToRunOnConfigDB = tmpOptions.allowToRunOnConfigDB;
        delete tmpOptions.allowToRunOnConfigDB;
    }

    if (tmpOptions.hasOwnProperty("allowToRunOnSystemNS")) {
        changeStreamStage.allowToRunOnSystemNS = tmpOptions.allowToRunOnSystemNS;
        delete tmpOptions.allowToRunOnSystemNS;
    }

    if (tmpOptions.hasOwnProperty("startAtOperationTime")) {
        changeStreamStage.startAtOperationTime = tmpOptions.startAtOperationTime;
        delete tmpOptions.startAtOperationTime;
    }

    if (options.hasOwnProperty("showExpandedEvents")) {
        changeStreamStage.showExpandedEvents = tmpOptions.showExpandedEvents;
        delete tmpOptions.showExpandedEvents;
    }

    if (options.hasOwnProperty("showSystemEvents")) {
        changeStreamStage.showSystemEvents = tmpOptions.showSystemEvents;
        delete tmpOptions.showSystemEvents;
    }

    if (options.hasOwnProperty("showRawUpdateDescription")) {
        changeStreamStage.showRawUpdateDescription = tmpOptions.showRawUpdateDescription;
        delete tmpOptions.showRawUpdateDescription;
    }

    tmpPipeline.unshift({ $changeStream: changeStreamStage });
    return this.aggregate(tmpPipeline, tmpOptions);
};

NAVCollection.prototype.checkMetadataConsistency = function (options = {}) {
    if (!isObject(options))
        nav_throwError("'options' parameter is " + typeof options + " but not object");

    var result = this.runCommand(Object.extend({ checkMetadataConsistency: this.getName() }, options));
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result.cursor.firstBatch;
};

NAVCollection.prototype.createSearchIndex = function (name, definition) {
    if (arguments.length > 2)
        nav_throwError("invalid args");

    if (isObject(name) && name != null) {
        definition = name;
        name = undefined;
    }

    if (isUndefined(definition) || !isObject(definition))
        nav_throwError("'definition' for collection.createSearchIndex should be object");

    var commandObj = { createSearchIndexes: this.getName() };
    var tmpObject = { definition: definition };
    if (isString(name))
        tmpObject.name = name;

    commandObj.indexes = [tmpObject];

    return this.runCommand(commandObj);
};

NAVCollection.prototype.dropSearchIndex = function (name) {
    if (!isString(name))
        nav_throwError("'name' for collection.dropSearchIndex should be string");

    var commandObj = { dropSearchIndex: this.getName(), name: name };
    return this.runCommand(commandObj);
};

NAVCollection.prototype.getSearchIndexes = function (name) {
    var pipeline = [{ "$listSearchIndexes": {} }];
    if (isString(name))
        pipeline[0].$listSearchIndexes.name = name;
    return this.aggregate(pipeline).toArray();
};

NAVCollection.prototype.updateSearchIndex = function (name, definition) {
    if (arguments.length > 2)
        nav_throwError("invalid args");

    if (isUndefined(name) || !isString(name))
        nav_throwError("'name' for collection.createSearchIndex should be string");
    if (isUndefined(definition) || !isObject(definition))
        nav_throwError("'definition' for collection.createSearchIndex should be object");

    var commandObj = {};
    commandObj.updateSearchIndex = this.getName();
    commandObj.definition = definition;
    commandObj.name = name;

    return this.runCommand(commandObj);
};

// NAVCollectionChainInfo
var DBQuery = {};
DBQuery.Option = {
    tailable: 0x2,
    slaveOk: 0x4,
    oplogReplay: 0x8,
    noTimeout: 0x10,
    awaitData: 0x20,
    exhaust: 0x40,
    partial: 0x80
};

NAVCollectionChainInfo.prototype.addOption = function (option) {
    if (option == DBQuery.Option.tailable)
        return this.forwardToCustomFunction("tailable");
    else if (option == DBQuery.Option.oplogReplay)
        return this.forwardToCustomFunction("oplogReplay");
    else if (option == DBQuery.Option.slaveOk)
        return this.forwardToCustomFunction("slaveOk");
    else if (option == DBQuery.Option.noTimeout)
        return this.forwardToCustomFunction("noCursorTimeout");
    else if (option == DBQuery.Option.awaitData)
        return this.forwardToCustomFunction("awaitData");
    else if (option == DBQuery.Option.exhaust)
        return this.forwardToCustomFunction("exhaust");
    else if (option == DBQuery.Option.partial)
        return this.forwardToCustomFunction("allowPartialResults");
};

NAVCollectionChainInfo.prototype.allowDiskUse = function (value) {
    value = isUndefined(value) ? true : value;
    return this.forwardToCustomFunction("allowDiskUse", value);
};

NAVCollectionChainInfo.prototype.batchSize = function (size) {
    return this.forwardToCustomFunction("batchSize", size);
};

NAVCollectionChainInfo.prototype.clone = function () {
    return this.forwardToCustomFunction("clone");
};

NAVCollectionChainInfo.prototype.close = function () {
    this.forwardToCustomFunction("close");
};

NAVCollectionChainInfo.prototype.collation = function (collation) {
    return this.forwardToCustomFunction("collation", collation);
};

NAVCollectionChainInfo.prototype.comment = function (comment) {
    return this.forwardToCustomFunction("comment", comment);
};

NAVCollectionChainInfo.prototype.count = function (applySkipLimit) {
    if (isUndefined(applySkipLimit))
        applySkipLimit = false;
    if (isNumber(applySkipLimit))
        applySkipLimit = (applySkipLimit !== 0);
    if (!isBoolean(applySkipLimit))
        applySkipLimit = true;

    return this.forwardToCustomFunction("count", applySkipLimit);
};

NAVCollectionChainInfo.prototype.explain = function (verbosity) {
    if (verbosity && !isString(verbosity)) 
        verbosity = "allPlansExecution";
    if (!verbosity && !isString(verbosity))
        verbosity = "queryPlanner";
    if (verbosity !== "queryPlanner" && verbosity !== "executionStats" && verbosity !== "allPlansExecution")
        nav_throwError("explain verbosity must be one of {'queryPlanner', 'executionStats', 'allPlansExecution'}")

    return this.forwardToCustomFunction("explain", verbosity);
};

NAVCollectionChainInfo.prototype.finish = function () {
    return this.forwardToCustomFunction("finish");
};

NAVCollectionChainInfo.prototype.forEach = function (func) {
    while (this.hasNext())
        func(this.next());
};

NAVCollectionChainInfo.prototype.hasNext = function () {
    return this.forwardToCustomFunction("hasNext");
};

NAVCollectionChainInfo.prototype.help = function () {
    print("find(<predicate>, <projection>) modifiers\n" +
        "\t.sort({...})\n" +
        "\t.limit(<n>)\n" +
        "\t.skip(<n>)\n" +
        "\t.batchSize(<n>) - sets the number of docs to return per getMore\n" +
        "\t.collation({...})\n" +
        "\t.hint({...})\n" +
        "\t.readConcern(<level>)\n" +
        "\t.readPref(<mode>, <tagset>)\n" +
        "\t.count(<applySkipLimit>) - total # of objects matching query. by default ignores skip,limit\n" +
        "\t.size() - total # of objects cursor would return, honors skip,limit\n" +
        "\t.explain(<verbosity>) - accepted verbosities are {'queryPlanner', 'executionStats', 'allPlansExecution'}\n" +
        "\t.min({...})\n" +
        "\t.max({...})\n" +
        "\t.maxTimeMS(<n>)\n" +
        "\t.maxAwaitTimeMS(<n>)\n" +
        "\t.comment(<comment>)\n" +
        "\t.tailable(<isAwaitData>)\n" +
        "\t.noCursorTimeout()\n" +
        "\t.allowPartialResults()\n" +
        "\t.returnKey()\n" +
        "\t.showRecordId() - adds a $recordId field to each returned object\n" +
        "\t.allowDiskUse() - allow using disk in completing the query\n" +

        "\nCursor methods\n" +
        "\t.toArray() - iterates through docs and returns an array of the results\n" +
        "\t.forEach(<func>)\n" +
        "\t.map(<func>)\n" +
        "\t.hasNext()\n" +
        "\t.next()\n" +
        "\t.tryNext()\n" +
        "\t.close()\n" +
        "\t.objsLeftInBatch() - returns count of docs left in current batch (when exhausted, a new getMore will be issued)\n" +
        "\t.itcount() - iterates through documents and counts them\n" +
        "\t.getClusterTime() - returns the read timestamp for snapshot reads\n" +
        "\t.pretty() - pretty print each document, possibly over multiple lines\n");
};

NAVCollectionChainInfo.prototype.hint = function (hint) {
    return this.forwardToCustomFunction("hint", hint);
};

NAVCollectionChainInfo.prototype.length = function () {
    return this.toArray().length;
};

NAVCollectionChainInfo.prototype.isClosed = function () {
    return this.forwardToCustomFunction("isClosed");
};

NAVCollectionChainInfo.prototype.isExhausted = function () {
    return this.forwardToCustomFunction("isExhausted");
};

NAVCollectionChainInfo.prototype.itcount = function () {
    return this.forwardToCustomFunction("itcount");
};

NAVCollectionChainInfo.prototype.limit = function (limit) {
    return this.forwardToCustomFunction("limit", limit);
};

NAVCollectionChainInfo.prototype.map = function (jsFunction) {
    var array = [];
    while (this.hasNext())
        array.push(jsFunction(this.next()));
    return array;
};

NAVCollectionChainInfo.prototype.max = function (max) {
    return this.forwardToCustomFunction("max", max);
};

NAVCollectionChainInfo.prototype.maxScan = function (maxScan) {
    return this.forwardToCustomFunction("maxScan", maxScan);
};

NAVCollectionChainInfo.prototype.maxAwaitTimeMS = function (timeLimit) {
    return this.forwardToCustomFunction("maxAwaitTimeMS", timeLimit);
};

NAVCollectionChainInfo.prototype.maxTimeMS = function (timeLimit) {
    return this.forwardToCustomFunction("maxTimeMS", timeLimit);
};

NAVCollectionChainInfo.prototype.min = function (min) {
    return this.forwardToCustomFunction("min", min);
};

NAVCollectionChainInfo.prototype.next = function () {
    return this.forwardToCustomFunction("next");
};

NAVCollectionChainInfo.prototype.noCursorTimeout = function () {
    return this.forwardToCustomFunction("noCursorTimeout");
};

NAVCollectionChainInfo.prototype.objsLeftInBatch = function () {
    return this.forwardToCustomFunction("objsLeftInBatch");
};

NAVCollectionChainInfo.prototype.pretty = function () {
    return this;
};

NAVCollectionChainInfo.prototype.readConcern = function (level) {
    return this.forwardToCustomFunction("readConcern", level);
};

NAVCollectionChainInfo.prototype.readPref = function (mode, tagSet, hedgeOptions) {
    return this.forwardToCustomFunction("readPref", mode, tagSet, hedgeOptions);
};

NAVCollectionChainInfo.prototype.returnKey = function () {
    return this.forwardToCustomFunction("returnKey");
};

NAVCollectionChainInfo.prototype.showRecordId = function () {
    return this.forwardToCustomFunction("showRecordId");
};
NAVCollectionChainInfo.prototype.showDiskLoc = NAVCollectionChainInfo.prototype.showRecordId;

NAVCollectionChainInfo.prototype.size = function () {
    return this.forwardToCustomFunction("size");
};

NAVCollectionChainInfo.prototype.skip = function (skip) {
    return this.forwardToCustomFunction("skip", skip);
};

NAVCollectionChainInfo.prototype.snapshot = function () {
    return this.forwardToCustomFunction("snapshot");
};

NAVCollectionChainInfo.prototype.sort = function (columns) {
    return this.forwardToCustomFunction("sort", columns);
};

NAVCollectionChainInfo.prototype.tailable = function (awaitData) {
    var isAwaitData = false;
    if (awaitData || awaitData == null)
        isAwaitData = true;

    if (isAwaitData)
        return this.forwardToCustomFunction("tailable", isAwaitData);
    else
        return this.forwardToCustomFunction("tailable");
};

NAVCollectionChainInfo.prototype.toArray = function () {
    if (this._array)
        return this._array;

    var newArray = [];
    while (this.hasNext())
        newArray.push(this.next());
    this._array = newArray;
    return newArray;
}

NAVCollectionChainInfo.prototype.tryNext = function () {
    return this.forwardToCustomFunction("tryNext");
};

// NAVCollectionExplain
NAVCollectionExplain.prototype.getCollection = function() {
    return this._collection;
};

NAVCollectionExplain.prototype.getVerbosity = function() {
    return this.forwardToCustomFunction("collectionExplainGetVerbosity");
};

NAVCollectionExplain.prototype.setVerbosity = function(verbosity) {
    if (verbosity && !isString(verbosity)) 
        verbosity = "allPlansExecution";
    if (!verbosity && !isString(verbosity))
        verbosity = "queryPlanner";
    if (verbosity !== "queryPlanner" && verbosity !== "executionStats" && verbosity !== "allPlansExecution")
        nav_throwError("explain verbosity must be one of {'queryPlanner', 'executionStats', 'allPlansExecution'}")

    this.forwardToCustomFunction("collectionExplainSetVerbosity", verbosity)
    return this;
};

NAVCollectionExplain.prototype.help = function () {
    print("Explainable operations\n" +
        "\t.aggregate(...) - explain an aggregation operation\n" +
        "\t.count(...) - explain a count operation\n" +
        "\t.distinct(...) - explain a distinct operation\n" +
        "\t.find(...) - get an explainable query\n" +
        "\t.findAndModify(...) - explain a findAndModify operation\n" +
        "\t.group(...) - explain a group operation\n" +
        "\t.remove(...) - explain a remove operation\n" +
        "\t.update(...) - explain an update operation\n" +
        "Explainable collection methods\n" +
        "\t.getCollection()\n" +
        "\t.getVerbosity()\n" +
        "\t.setVerbosity(verbosity)\n");
};

NAVCollectionExplain.prototype.aggregate = function (pipeline, options) {
    if (arguments.length == 0)
        pipeline = [];
    
    if (!Array.isArray(pipeline))
        nav_throwError("invalid argument");

    if (isUndefined(options))
        options = {};
    
    if (!isObject(options))
        nav_throwError("invalid argument");

    var tmpOptions = Object.extend({}, (options || {}));

    if (!("cursor" in tmpOptions))
        tmpOptions.cursor = {};

    if (this.getVerbosity() === "queryPlanner") {
        tmpOptions.explain = true;
        return this._collection.aggregate(pipeline, tmpOptions);
    } else {
        // suppose version >= 3.5
        var commandObj = { aggregate : this._collection.getName(), pipeline : pipeline};
        Object.extend(commandObj, tmpOptions);
        commandObj = { explain : commandObj, verbosity : this.getVerbosity() }

        return this.forwardToCustomFunction("collectionExplainRunReadCommand", commandObj);
    }
};

NAVCollectionExplain.prototype.count = function(query, options) {
    return this.find(query, options).count();
};

NAVCollectionExplain.prototype.distinct = function (field, query, options) {
    var commandObj = {
        distinct: this._collection.getName(),
        key: field,
        query: query || {}
    };

    if (options && options.hasOwnProperty("collation")) {
        commandObj.collation = options.collation;
    }

    commandObj = { explain: commandObj, verbosity: this.getVerbosity() };
    return this.forwardToCustomFunction("collectionExplainRunReadCommand", commandObj);
};

NAVCollectionExplain.prototype.find = function (query, projection, options) {
    if (this._collection._db.getDatabaseVersion() >= 50000) {
        return this.forwardToCustomFunction("collectionExplainFind", query, projection, options);
    } else {
        return this.forwardToCustomFunction("collectionExplainFind", query, projection);
    }
};

NAVCollectionExplain.prototype.findAndModify = function(params) {
    var commandObj = Object.extend({"findAndModify": this._collection.getName()}, params);
    commandObj = {"explain": commandObj, "verbosity": this.getVerbosity()};
    return this.forwardToCustomFunction("collectionExplainRunReadCommand", commandObj);
};

NAVCollectionExplain.prototype.group = function(params) {
    var _groupFixParms = function (parmsobj) {
        var parms = Object.extend({}, parmsobj);

        if (parms.reduce) {
            parms.$reduce = parms.reduce;
            delete parms.reduce;
        }

        if (parms.keyf) {
            parms.$keyf = parms.keyf;
            delete parms.keyf;
        }

        return parms;
    };

    params.ns = this._collection.getName();
    var commandObj = {group: _groupFixParms(params)};
    commandObj = {explain: commandObj, verbosity: this.getVerbosity()};
    return this.forwardToCustomFunction("collectionExplainRunReadCommand", commandObj);
};

function _tranlateObject(query) {
    if (!query)
        return {};

    if (isFunction(query))
        return {$where: query};

    if (query.isObjectId)
        return {_id: query};

    if (isObject(query))
        return query;

    if (isString(query)) {
        if (/^[0-9a-fA-F]{24}$/.test(query))
            return {_id: ObjectId(query)};
        return {$where: query};
    }

    nav_throwError("unknown object to be translated");

};

function _parseRemove(query, justOne) {
    if (isUndefined(query))
        nav_throwError("remove needs a query");

    query = _tranlateObject(query);

    var writeConcern = undefined;
    var collation = undefined;
    if (isObject(justOne)) {
        var options = justOne;
        writeConcern = options.writeConcern;
        justOne = options.justOne;
        collation = options.collation;
        letParams = options.let;
    }

    justOne = justOne ? true : false;

    return { query: query, justOne: justOne, wc: writeConcern, collation: collation, let: letParams };
};

NAVCollectionExplain.prototype.remove = function(query, justOne) {
    var parsed = _parseRemove(query, justOne);
    query = parsed.query;
    justOne = parsed.justOne;
    var collation = parsed.collation;
    var letParams = parsed.let;

    var commandObj = { q : query, limit : justOne}
    if (collation) {
        commandObj.collation = collation;
    }

    commandObj = { explain : { delete : this._collection.getName(), deletes : [commandObj], ordered : true}, verbosity : this.getVerbosity()};

    if (letParams) {
        commandObj.explain.let = letParams;
    }

    return this.forwardToCustomFunction("collectionExplainRunCommand", commandObj);
};

function _parseUpdate(query, updateSpec, upsert, multi) {
    if (!query)
        nav_throwError("need a query");
    if (!updateSpec)
        nav_throwError("need an update object or pipeline");

    var wc = undefined;
    var collation = undefined;
    var arrayFilters = undefined;
    var hint = undefined;
    var letParams = undefined;
    if (isObject(upsert)) {
        if (multi)
            nav_throwError("Fourth argument must be empty when specifying " +
                        "upsert and multi with an object.");

        var options = upsert;
        multi = options.multi;
        wc = options.writeConcern;
        upsert = options.upsert;
        collation = options.collation;
        arrayFilters = options.arrayFilters;
        hint = options.hint;
        letParams = options.let;
    }

    upsert = upsert ? true : false;
    multi = multi ? true : false;

    return {
        "query": query,
        "updateSpec": updateSpec,
        "hint": hint,
        "upsert": upsert,
        "multi": multi,
        "wc": wc,
        "collation": collation,
        "arrayFilters": arrayFilters,
        "let": letParams
    };
};

NAVCollectionExplain.prototype.update = function(query, updateSpec, upsert, multi) {
    var parsed = _parseUpdate(query, updateSpec, upsert, multi);
    var query = parsed.query;
    var updateSpec = parsed.updateSpec;
    var upsert = parsed.upsert;
    var multi = parsed.multi;
    var collation = parsed.collation;
    var arrayFilters = parsed.arrayFilters;
    var hint = parsed.hint;
    var letParams = parsed.let;

    var commandObj = { q: query, u: updateSpec, upsert: upsert, multi: multi }
    if (collation) {
        commandObj.collation = collation;
    }
    if (arrayFilters) {
        commandObj.arrayFilters = arrayFilters;
    }
    if (hint) {
        commandObj.hint = hint;
    }

    commandObj = { explain : { update : this._collection.getName(), updates : [commandObj], ordered : true}, verbosity : this.getVerbosity()};

    if (letParams) {
        commandObj.explain.let = letParams;
    }
    
    return this.forwardToCustomFunction("collectionExplainRunCommand", commandObj);
};
